// Side Panel Chat UI

const messagesContainer = document.getElementById('messages');
const inputField = document.getElementById('input');
const sendButton = document.getElementById('send-btn');
const clearButton = document.getElementById('clear-btn');
const statusBar = document.getElementById('status-bar');
const statusText = statusBar.querySelector('.status-text');

let isProcessing = false;
let port = null;
let pingInterval = null;

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  loadHistory();
  setupEventListeners();
  setupPortConnection();
});

// Setup persistent port connection to keep Service Worker alive
function setupPortConnection() {
  try {
    port = chrome.runtime.connect({ name: 'keepalive' });
    console.log('Port connected to background');

    port.onMessage.addListener((msg) => {
      if (msg.type === 'pong') {
        if (msg.connected) {
          hideStatus();
        } else {
          showStatus('disconnected', 'Подключение...');
        }
      } else {
        handleBackgroundMessage(msg);
      }
    });

    port.onDisconnect.addListener(() => {
      console.log('Port disconnected, reconnecting...');
      port = null;
      setTimeout(setupPortConnection, 1000);
    });

    // Send ping every 20 seconds to keep connection alive
    if (pingInterval) clearInterval(pingInterval);
    pingInterval = setInterval(() => {
      if (port) {
        port.postMessage({ type: 'ping' });
      }
    }, 20000);

    // Initial ping
    port.postMessage({ type: 'ping' });

  } catch (e) {
    console.error('Failed to connect port:', e);
    setTimeout(setupPortConnection, 2000);
  }
}

function setupEventListeners() {
  // Send on button click
  sendButton.addEventListener('click', sendMessage);

  // Send on Enter (but not Shift+Enter)
  inputField.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  });

  // Auto-resize textarea
  inputField.addEventListener('input', () => {
    inputField.style.height = 'auto';
    inputField.style.height = Math.min(inputField.scrollHeight, 120) + 'px';
    sendButton.disabled = !inputField.value.trim() || isProcessing;
  });

  // Clear history
  clearButton.addEventListener('click', clearHistory);

  // Suggestion clicks
  document.addEventListener('click', (e) => {
    if (e.target.matches('.suggestions li')) {
      const query = e.target.dataset.query;
      inputField.value = query;
      sendMessage();
    }
  });

  // Listen for messages from background
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    handleBackgroundMessage(message);
  });
}

function sendMessage() {
  const text = inputField.value.trim();
  if (!text || isProcessing) return;

  // Clear welcome message
  const welcome = document.querySelector('.welcome-message');
  if (welcome) welcome.remove();

  // Add user message to UI
  addMessage('user', text);

  // Clear input
  inputField.value = '';
  inputField.style.height = 'auto';
  sendButton.disabled = true;

  // Send to background via runtime message
  chrome.runtime.sendMessage({
    type: 'user_message',
    text: text
  });

  setProcessing(true);
}

function handleBackgroundMessage(message) {
  switch (message.type) {
    case 'ai_response':
      addMessage('assistant', message.text);
      setProcessing(false);
      hideStatus();
      break;

    case 'ai_stream':
      // Handle streaming chunk
      updateStreamingMessage(message.text);
      break;

    case 'ai_status':
      showStatus(message.status);
      break;

    case 'ai_error':
      addMessage('error', message.error);
      setProcessing(false);
      hideStatus();
      break;

    case 'connection_status':
      if (message.connected) {
        hideStatus();
      } else {
        showStatus('disconnected', 'Нет соединения с сервером');
      }
      break;
  }
}

function addMessage(role, content) {
  // Remove typing indicator if present
  const typing = document.querySelector('.typing-indicator');
  if (typing) typing.parentElement.remove();

  const messageDiv = document.createElement('div');
  messageDiv.className = `message ${role}`;

  const contentDiv = document.createElement('div');
  contentDiv.className = 'message-content';
  contentDiv.innerHTML = formatMessage(content);

  messageDiv.appendChild(contentDiv);
  messagesContainer.appendChild(messageDiv);

  scrollToBottom();
  saveHistory();
}

function updateStreamingMessage(chunk) {
  let streamingMsg = document.querySelector('.message.streaming');

  if (!streamingMsg) {
    streamingMsg = document.createElement('div');
    streamingMsg.className = 'message assistant streaming';

    const contentDiv = document.createElement('div');
    contentDiv.className = 'message-content';
    streamingMsg.appendChild(contentDiv);

    messagesContainer.appendChild(streamingMsg);
  }

  const content = streamingMsg.querySelector('.message-content');
  content.textContent += chunk;

  scrollToBottom();
}

function finalizeStreamingMessage() {
  const streamingMsg = document.querySelector('.message.streaming');
  if (streamingMsg) {
    streamingMsg.classList.remove('streaming');
    const content = streamingMsg.querySelector('.message-content');
    content.innerHTML = formatMessage(content.textContent);
    saveHistory();
  }
}

function formatMessage(text) {
  // Simple markdown-like formatting
  return text
    .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
    .replace(/\*(.*?)\*/g, '<em>$1</em>')
    .replace(/`(.*?)`/g, '<code>$1</code>')
    .replace(/\n/g, '<br>')
    .replace(/• /g, '&bull; ');
}

function showStatus(status, customText) {
  statusBar.classList.remove('hidden', 'thinking', 'executing', 'error');

  switch (status) {
    case 'thinking':
      statusBar.classList.add('thinking');
      statusText.textContent = 'AI думает...';
      break;
    case 'executing_tools':
      statusBar.classList.add('executing');
      statusText.textContent = 'Выполняется запрос к Ленте...';
      break;
    case 'disconnected':
      statusBar.classList.add('error');
      statusText.textContent = customText || 'Нет соединения';
      break;
    default:
      statusText.textContent = customText || status;
  }
}

function hideStatus() {
  statusBar.classList.add('hidden');
}

function setProcessing(processing) {
  isProcessing = processing;
  sendButton.disabled = processing || !inputField.value.trim();

  if (processing) {
    // Show typing indicator
    const typingDiv = document.createElement('div');
    typingDiv.className = 'message assistant';
    typingDiv.innerHTML = `
      <div class="message-content">
        <div class="typing-indicator">
          <span></span>
          <span></span>
          <span></span>
        </div>
      </div>
    `;
    messagesContainer.appendChild(typingDiv);
    scrollToBottom();
  }
}

function scrollToBottom() {
  messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

function saveHistory() {
  const messages = Array.from(document.querySelectorAll('.message:not(.streaming)'))
    .map(msg => {
      const contentEl = msg.querySelector('.message-content');
      // Skip messages without content (like typing indicators)
      if (!contentEl || contentEl.querySelector('.typing-indicator')) return null;
      return {
        role: msg.classList.contains('user') ? 'user' : 'assistant',
        content: contentEl.textContent
      };
    })
    .filter(Boolean);

  chrome.storage.local.set({ chatHistory: messages });
}

function loadHistory() {
  chrome.storage.local.get(['chatHistory'], (result) => {
    if (result.chatHistory && result.chatHistory.length > 0) {
      // Remove welcome message
      const welcome = document.querySelector('.welcome-message');
      if (welcome) welcome.remove();

      result.chatHistory.forEach(msg => {
        addMessage(msg.role, msg.content);
      });
    }
  });
}

function clearHistory() {
  chrome.storage.local.remove(['chatHistory']);
  messagesContainer.innerHTML = `
    <div class="welcome-message">
      <h2>Привет! 👋</h2>
      <p>Я помогу тебе с покупками в Ленте.</p>
      <p>Попробуй спросить:</p>
      <ul class="suggestions">
        <li data-query="Найди молоко дешевле 100 рублей">Найди молоко дешевле 100₽</li>
        <li data-query="Какой сыр лучше по соотношению цена/качество?">Какой сыр лучше?</li>
        <li data-query="Собери корзину для борща на 4 порции">Собери корзину для борща</li>
      </ul>
    </div>
  `;

  // Also clear server-side history
  chrome.runtime.sendMessage({ type: 'clear_history' });
}
