// Content Script - Lenta API Client
// Runs in context of lenta.com pages with access to cookies

const LENTA_BASE = 'https://lenta.com';
const CLIENT_VERSION = 'angular_web_0.0.2';

// Get session data from cookies and localStorage
function getSessionData() {
  const getCookie = (name) => {
    const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
    return match ? decodeURIComponent(match[2]) : null;
  };

  return {
    deviceId: getCookie('Utk_DvcGuid') || localStorage.getItem('deviceId') || generateUUID(),
    sessionToken: getCookie('Utk_SessionToken') || '',
    userSessionId: getCookie('UserSessionId') || generateUUID(),
    domain: getCookie('ReferenceRegionName') || 'moscow',
  };
}

function generateUUID() {
  return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
    const r = Math.random() * 16 | 0;
    const v = c === 'x' ? r : (r & 0x3 | 0x8);
    return v.toString(16);
  });
}

// Common headers for Lenta API
function getHeaders(contentType = 'application/json') {
  const session = getSessionData();

  return {
    'Accept': 'application/json',
    'Content-Type': contentType,
    'Client': CLIENT_VERSION,
    'X-Retail-Brand': 'lo',
    'X-Domain': session.domain,
    'X-Delivery-Mode': 'delivery',
    'X-Platform': 'omniweb',
    'Deviceid': session.deviceId,
    'Sessiontoken': session.sessionToken,
    'X-Device-Id': session.deviceId,
    'X-User-Session-Id': session.userSessionId,
  };
}

// Tool Handlers

const toolHandlers = {
  // Search products by query
  search_products: async ({ query, limit = 20 }) => {
    const response = await fetch(`${LENTA_BASE}/api-gateway/v1/catalog/items`, {
      method: 'POST',
      headers: getHeaders(),
      credentials: 'include',
      body: JSON.stringify({
        searchText: query,
        filters: { checkbox: [], multicheckbox: [], range: [] },
        sort: { type: 'popular', order: 'desc' },
        limit: limit,
        offset: 0
      })
    });

    if (!response.ok) {
      throw new Error(`Search failed: ${response.status}`);
    }

    const data = await response.json();

    // Transform to simplified format for AI
    return {
      total: data.total || 0,
      products: (data.items || []).map(item => ({
        sku: item.sku || item.skuCode,
        name: item.title || item.name,
        price: item.prices?.price || item.price,
        oldPrice: item.prices?.oldPrice || item.oldPrice,
        unit: item.unitOfMeasure,
        weight: item.weight,
        brand: item.brand,
        image: item.image?.small || item.imageUrl,
        inStock: item.inStock !== false,
        promo: item.promoLabel || null
      }))
    };
  },

  // Get product details by SKU
  get_product: async ({ sku }) => {
    const response = await fetch(`${LENTA_BASE}/api-gateway/v1/catalog/sku/${sku}`, {
      method: 'GET',
      headers: getHeaders(),
      credentials: 'include'
    });

    if (!response.ok) {
      throw new Error(`Product not found: ${response.status}`);
    }

    const item = await response.json();

    return {
      sku: item.sku || item.skuCode,
      name: item.title || item.name,
      description: item.description,
      price: item.prices?.price || item.price,
      oldPrice: item.prices?.oldPrice || item.oldPrice,
      unit: item.unitOfMeasure,
      weight: item.weight,
      brand: item.brand,
      manufacturer: item.manufacturer,
      country: item.country,
      composition: item.composition,
      nutritionFacts: item.nutritionFacts,
      image: item.image?.large || item.imageUrl,
      inStock: item.inStock !== false
    };
  },

  // Get categories tree
  get_categories: async () => {
    const timestamp = Date.now();
    const response = await fetch(`${LENTA_BASE}/api-gateway/v1/catalog/categories?timestamp=${timestamp}`, {
      method: 'GET',
      headers: getHeaders(),
      credentials: 'include'
    });

    if (!response.ok) {
      throw new Error(`Failed to get categories: ${response.status}`);
    }

    const data = await response.json();

    // Flatten and simplify categories
    const simplifyCategories = (categories, parentName = '') => {
      const result = [];
      for (const cat of (categories || [])) {
        result.push({
          id: cat.id,
          name: cat.name,
          slug: cat.slug,
          parent: parentName || null
        });
        if (cat.children && cat.children.length > 0) {
          result.push(...simplifyCategories(cat.children, cat.name));
        }
      }
      return result;
    };

    return {
      categories: simplifyCategories(data.categories || data)
    };
  },

  // Get products from category
  get_category_products: async ({ category_id, limit = 20, offset = 0, sort = 'popular' }) => {
    const sortMap = {
      'popular': { type: 'popular', order: 'desc' },
      'price_asc': { type: 'price', order: 'asc' },
      'price_desc': { type: 'price', order: 'desc' },
      'discount': { type: 'discount', order: 'desc' }
    };

    const response = await fetch(`${LENTA_BASE}/api-gateway/v1/catalog/items`, {
      method: 'POST',
      headers: getHeaders(),
      credentials: 'include',
      body: JSON.stringify({
        categoryId: category_id,
        filters: { checkbox: [], multicheckbox: [], range: [] },
        sort: sortMap[sort] || sortMap.popular,
        limit: limit,
        offset: offset
      })
    });

    if (!response.ok) {
      throw new Error(`Category products failed: ${response.status}`);
    }

    const data = await response.json();

    return {
      total: data.total || 0,
      products: (data.items || []).map(item => ({
        sku: item.sku || item.skuCode,
        name: item.title || item.name,
        price: item.prices?.price || item.price,
        oldPrice: item.prices?.oldPrice || item.oldPrice,
        unit: item.unitOfMeasure,
        weight: item.weight,
        brand: item.brand,
        inStock: item.inStock !== false,
        promo: item.promoLabel || null
      }))
    };
  },

  // Get cart contents
  get_cart: async () => {
    const session = getSessionData();

    const requestBody = {
      Head: {
        MarketingPartnerKey: 'mp300-b1de0bac2c257f3257bf5ef2eea4ecbc',
        Version: CLIENT_VERSION,
        Client: CLIENT_VERSION,
        Method: 'cartLookup',
        RequestId: `cart_${Date.now()}`,
        DeviceId: session.deviceId,
        Domain: session.domain,
        SessionToken: session.sessionToken
      },
      Body: {
        Return: { Goods: 1 }
      }
    };

    const response = await fetch(`${LENTA_BASE}/api/rest/cartLookup`, {
      method: 'POST',
      headers: {
        ...getHeaders('application/x-www-form-urlencoded'),
      },
      credentials: 'include',
      body: `request=${encodeURIComponent(JSON.stringify(requestBody))}`
    });

    if (!response.ok) {
      throw new Error(`Cart lookup failed: ${response.status}`);
    }

    const data = await response.json();

    // Parse cart response
    const cart = data.Body?.Cart || data.cart || {};
    const goods = cart.Goods || cart.goods || [];

    return {
      itemCount: goods.length,
      totalPrice: cart.Total || cart.total || 0,
      items: goods.map(item => ({
        sku: item.SkuCode || item.sku,
        name: item.Title || item.name,
        price: item.Price || item.price,
        quantity: item.Quantity || item.quantity,
        subtotal: (item.Price || item.price) * (item.Quantity || item.quantity)
      }))
    };
  },

  // Get promotions
  get_promotions: async ({ limit = 20 }) => {
    const response = await fetch(`${LENTA_BASE}/api-gateway/v1/catalog/items`, {
      method: 'POST',
      headers: getHeaders(),
      credentials: 'include',
      body: JSON.stringify({
        filters: { checkbox: [], multicheckbox: [], range: [] },
        sort: { type: 'discount', order: 'desc' },
        limit: limit,
        offset: 0,
        promoOnly: true
      })
    });

    if (!response.ok) {
      throw new Error(`Promotions failed: ${response.status}`);
    }

    const data = await response.json();

    return {
      total: data.total || 0,
      promotions: (data.items || []).map(item => ({
        sku: item.sku || item.skuCode,
        name: item.title || item.name,
        price: item.prices?.price || item.price,
        oldPrice: item.prices?.oldPrice || item.oldPrice,
        discount: item.prices?.oldPrice
          ? Math.round((1 - (item.prices?.price || item.price) / item.prices.oldPrice) * 100)
          : 0,
        promo: item.promoLabel || null
      }))
    };
  },

  // Add item to cart
  add_to_cart: async ({ sku, quantity = 1 }) => {
    const session = getSessionData();

    const requestBody = {
      Head: {
        MarketingPartnerKey: 'mp300-b1de0bac2c257f3257bf5ef2eea4ecbc',
        Version: CLIENT_VERSION,
        Client: CLIENT_VERSION,
        Method: 'cartAdd',
        RequestId: `add_${Date.now()}`,
        DeviceId: session.deviceId,
        Domain: session.domain,
        SessionToken: session.sessionToken
      },
      Body: {
        SkuCode: sku,
        Quantity: quantity
      }
    };

    const response = await fetch(`${LENTA_BASE}/api/rest/cartAdd`, {
      method: 'POST',
      headers: {
        ...getHeaders('application/x-www-form-urlencoded'),
      },
      credentials: 'include',
      body: `request=${encodeURIComponent(JSON.stringify(requestBody))}`
    });

    if (!response.ok) {
      throw new Error(`Add to cart failed: ${response.status}`);
    }

    const data = await response.json();

    return {
      success: data.Body?.Success !== false,
      message: data.Body?.Message || 'Товар добавлен в корзину',
      cartItemCount: data.Body?.Cart?.ItemCount || null
    };
  },

  // Remove item from cart
  remove_from_cart: async ({ sku }) => {
    const session = getSessionData();

    const requestBody = {
      Head: {
        MarketingPartnerKey: 'mp300-b1de0bac2c257f3257bf5ef2eea4ecbc',
        Version: CLIENT_VERSION,
        Client: CLIENT_VERSION,
        Method: 'cartDelete',
        RequestId: `del_${Date.now()}`,
        DeviceId: session.deviceId,
        Domain: session.domain,
        SessionToken: session.sessionToken
      },
      Body: {
        SkuCode: sku
      }
    };

    const response = await fetch(`${LENTA_BASE}/api/rest/cartDelete`, {
      method: 'POST',
      headers: {
        ...getHeaders('application/x-www-form-urlencoded'),
      },
      credentials: 'include',
      body: `request=${encodeURIComponent(JSON.stringify(requestBody))}`
    });

    if (!response.ok) {
      throw new Error(`Remove from cart failed: ${response.status}`);
    }

    const data = await response.json();

    return {
      success: data.Body?.Success !== false,
      message: data.Body?.Message || 'Товар удален из корзины'
    };
  },

  // Update cart quantity
  update_cart_quantity: async ({ sku, quantity }) => {
    const session = getSessionData();

    const requestBody = {
      Head: {
        MarketingPartnerKey: 'mp300-b1de0bac2c257f3257bf5ef2eea4ecbc',
        Version: CLIENT_VERSION,
        Client: CLIENT_VERSION,
        Method: 'cartUpdate',
        RequestId: `upd_${Date.now()}`,
        DeviceId: session.deviceId,
        Domain: session.domain,
        SessionToken: session.sessionToken
      },
      Body: {
        SkuCode: sku,
        Quantity: quantity
      }
    };

    const response = await fetch(`${LENTA_BASE}/api/rest/cartUpdate`, {
      method: 'POST',
      headers: {
        ...getHeaders('application/x-www-form-urlencoded'),
      },
      credentials: 'include',
      body: `request=${encodeURIComponent(JSON.stringify(requestBody))}`
    });

    if (!response.ok) {
      throw new Error(`Update cart failed: ${response.status}`);
    }

    const data = await response.json();

    return {
      success: data.Body?.Success !== false,
      message: data.Body?.Message || 'Количество обновлено'
    };
  }
};

// Message handler for tool execution requests

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'execute_tool') {
    const { toolName, params, callId } = message;

    console.log('[Lenta Helper] Executing tool:', toolName, params);

    const handler = toolHandlers[toolName];
    if (!handler) {
      sendResponse({ success: false, error: `Unknown tool: ${toolName}` });
      return true;
    }

    // Execute handler async
    handler(params)
      .then(data => {
        console.log('[Lenta Helper] Tool result:', toolName, data);
        sendResponse({ success: true, data: JSON.stringify(data) });
      })
      .catch(error => {
        console.error('[Lenta Helper] Tool error:', toolName, error);
        sendResponse({ success: false, error: error.message });
      });

    return true; // Keep channel open for async response
  }
});

// Log that content script is loaded
console.log('[Lenta Helper] Content script loaded');
