// Background Service Worker
// Manages WebSocket connection and message routing

const SERVER_URL = 'wss://lenta-dev.dev.0sk.ru/ws';
// const SERVER_URL = 'ws://localhost:8080/ws'; // For local development

let ws = null;
let reconnectAttempts = 0;
const MAX_RECONNECT_ATTEMPTS = 10;
const RECONNECT_BASE_DELAY = 1000;
const KEEPALIVE_INTERVAL = 20000; // 20 seconds

// Track active Lenta tabs for tool execution
let activeLentaTabId = null;
let keepAliveTimer = null;

// Port connections from side panel - keeps SW alive
const ports = new Set();

// Initialize on install/startup
chrome.runtime.onInstalled.addListener(() => {
  console.log('Lenta Helper installed');
  connect();
});

chrome.runtime.onStartup.addListener(() => {
  console.log('Lenta Helper started');
  connect();
});

// Port connection from side panel - keeps Service Worker alive
chrome.runtime.onConnect.addListener((port) => {
  if (port.name === 'keepalive') {
    console.log('Side panel connected via port');
    ports.add(port);

    // Ensure WebSocket is connected
    if (!ws || ws.readyState !== WebSocket.OPEN) {
      connect();
    }

    port.onDisconnect.addListener(() => {
      console.log('Side panel disconnected');
      ports.delete(port);
    });

    // Forward messages from port
    port.onMessage.addListener((msg) => {
      if (msg.type === 'ping') {
        port.postMessage({ type: 'pong', connected: ws && ws.readyState === WebSocket.OPEN });
      }
    });
  }
});

// Keep Service Worker alive
function setupKeepAlive() {
  if (keepAliveTimer) clearInterval(keepAliveTimer);
  keepAliveTimer = setInterval(() => {
    // Ping to keep SW alive and check WS connection
    if (!ws || ws.readyState !== WebSocket.OPEN) {
      console.log('Keep-alive: reconnecting...');
      connect();
    }
  }, KEEPALIVE_INTERVAL);
}

// Open side panel when extension icon clicked
chrome.action.onClicked.addListener(async (tab) => {
  // Ensure connection before opening panel
  if (!ws || ws.readyState !== WebSocket.OPEN) {
    connect();
  }

  if (chrome.sidePanel && chrome.sidePanel.open) {
    try {
      await chrome.sidePanel.open({ tabId: tab.id });
    } catch (e) {
      console.error('Failed to open side panel:', e);
    }
  }
});

// Track active Lenta tabs
chrome.tabs.onActivated.addListener(async (activeInfo) => {
  try {
    const tab = await chrome.tabs.get(activeInfo.tabId);
    if (tab.url && tab.url.includes('lenta.com')) {
      activeLentaTabId = activeInfo.tabId;
    }
  } catch (e) {
    // Tab might be closed
  }
});

chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && tab.url && tab.url.includes('lenta.com')) {
    activeLentaTabId = tabId;
  }
});

// WebSocket Connection Management

function connect() {
  if (ws && ws.readyState === WebSocket.OPEN) {
    return;
  }

  console.log('Connecting to server...');

  try {
    ws = new WebSocket(SERVER_URL);

    ws.onopen = () => {
      console.log('Connected to server');
      reconnectAttempts = 0;
      notifyConnectionStatus(true);
    };

    ws.onmessage = (event) => {
      try {
        const message = JSON.parse(event.data);
        handleServerMessage(message);
      } catch (e) {
        console.error('Failed to parse server message:', e);
      }
    };

    ws.onerror = (error) => {
      console.error('WebSocket error:', error);
    };

    ws.onclose = (event) => {
      console.log('Disconnected from server:', event.code, event.reason);
      ws = null;
      notifyConnectionStatus(false);
      scheduleReconnect();
    };
  } catch (e) {
    console.error('Failed to create WebSocket:', e);
    scheduleReconnect();
  }
}

function scheduleReconnect() {
  if (reconnectAttempts >= MAX_RECONNECT_ATTEMPTS) {
    console.log('Max reconnect attempts reached');
    return;
  }

  const delay = Math.min(RECONNECT_BASE_DELAY * Math.pow(2, reconnectAttempts), 30000);
  reconnectAttempts++;

  console.log(`Reconnecting in ${delay}ms (attempt ${reconnectAttempts})`);
  setTimeout(connect, delay);
}

function sendToServer(message) {
  if (ws && ws.readyState === WebSocket.OPEN) {
    ws.send(JSON.stringify(message));
    return true;
  }
  console.error('WebSocket not connected');
  return false;
}

// Handle messages from server

function handleServerMessage(message) {
  console.log('Server message:', message.type, message);

  switch (message.type) {
    case 'response':
      // Final AI response
      sendToSidePanel({
        type: 'ai_response',
        text: message.text
      });
      break;

    case 'stream':
      // Streaming chunk
      sendToSidePanel({
        type: 'ai_stream',
        text: message.text
      });
      break;

    case 'status':
      // Status update (thinking, executing_tools, etc.)
      sendToSidePanel({
        type: 'ai_status',
        status: message.status
      });
      break;

    case 'error':
      // Error from server
      sendToSidePanel({
        type: 'ai_error',
        error: message.error
      });
      break;

    case 'command':
      // Tool execution command
      executeToolCommand(message);
      break;

    default:
      console.log('Unknown message type:', message.type);
  }
}

// Execute tool command via content script

async function executeToolCommand(command) {
  console.log('Executing tool:', command.action, command.params);

  // Find active Lenta tab
  const tabId = await findLentaTab();
  if (!tabId) {
    sendToolResult(command.id, {
      success: false,
      error: 'Откройте сайт lenta.com для выполнения запроса'
    });
    return;
  }

  try {
    // Send command to content script
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'execute_tool',
      toolName: command.action,
      params: JSON.parse(command.params),
      callId: command.id
    });

    sendToolResult(command.id, response);
  } catch (e) {
    console.error('Tool execution error:', e);
    sendToolResult(command.id, {
      success: false,
      error: e.message || 'Ошибка выполнения команды'
    });
  }
}

async function findLentaTab() {
  // First check if activeLentaTabId is still valid
  if (activeLentaTabId) {
    try {
      const tab = await chrome.tabs.get(activeLentaTabId);
      if (tab.url && tab.url.includes('lenta.com')) {
        return activeLentaTabId;
      }
    } catch (e) {
      activeLentaTabId = null;
    }
  }

  // Search for any Lenta tab
  const tabs = await chrome.tabs.query({ url: '*://lenta.com/*' });
  if (tabs.length > 0) {
    activeLentaTabId = tabs[0].id;
    return activeLentaTabId;
  }

  // Try with wildcard subdomain
  const tabs2 = await chrome.tabs.query({ url: '*://*.lenta.com/*' });
  if (tabs2.length > 0) {
    activeLentaTabId = tabs2[0].id;
    return activeLentaTabId;
  }

  return null;
}

function sendToolResult(callId, result) {
  sendToServer({
    type: 'tool_result',
    id: callId,
    success: result.success,
    data: result.data,
    error: result.error
  });
}

// Handle messages from side panel and content scripts

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Extension message:', message.type, message);

  switch (message.type) {
    case 'user_message':
      // User sent a message from side panel
      const sent = sendToServer({
        type: 'message',
        text: message.text
      });
      if (!sent) {
        sendToSidePanel({
          type: 'ai_error',
          error: 'Нет соединения с сервером'
        });
      }
      break;

    case 'clear_history':
      // User cleared chat history
      // Note: Server-side history will be cleared on next message
      break;

    case 'tool_result':
      // Content script returned tool result
      sendToolResult(message.callId, message.result);
      break;

    case 'get_connection_status':
      sendResponse({
        connected: ws && ws.readyState === WebSocket.OPEN
      });
      return true;

    case 'reconnect':
      // Force reconnect
      reconnectAttempts = 0;
      if (ws) {
        try { ws.close(); } catch (e) {}
        ws = null;
      }
      connect();
      sendResponse({ reconnecting: true });
      return true;
  }
});

// Send message to side panel

function sendToSidePanel(message) {
  // Send via ports (more reliable)
  for (const port of ports) {
    try {
      port.postMessage(message);
    } catch (e) {
      ports.delete(port);
    }
  }
  // Also try runtime message as fallback
  chrome.runtime.sendMessage(message).catch(() => {});
}

// Notify about connection status

function notifyConnectionStatus(connected) {
  sendToSidePanel({
    type: 'connection_status',
    connected: connected
  });
}

// Initial connection
connect();
